/**
 * @file This file provides some functions to handle registered configurations of the Information Center.
 * @description Do not modify this file. The file might reach release status.
 */

/* global sc */
/* jshint unused: true */


let metadb = require('common/metadb');


/**
 * The MetaDB path of the configurations.
 * @constant
 */
let METADB_SETTINGS_FOLDER_PATH = 'MAIN_MODULE_MANAGEMENT/customers/agorum.composite.acic/settings';



/**
 * Returns a list of configurations, optionally filtered by group name.
 * Typically the group name represents the name of the customer configuration, created by the template manager.
 * @param {string=} [groupName] - Optional. Name of the group.
 * @returns {object} The MetaDB entry containging the name, location of the corresponding JavaScript file and the ACL name.
 */
function listSettings(groupName) {
  let folderPath = METADB_SETTINGS_FOLDER_PATH;
  if (groupName) {
    folderPath += '/' + groupName;
  }
  return metadb.readJson(folderPath);
}



/**
 * List all configurations identified by it's setting name.
 * @param {string} settingName - The name of the setting.
 * @param {object} [user] - Optional. The object of the user context. Current logged in user is default.
 * @returns {object[]} An Array of configurations defined in the requrested setting.
 */
function listConfigurations(settingName, user) {
  let data = listSettings();
  let requestedSetting = data[settingName];
  let reqiredSetting = require(requestedSetting['js']);
  if (!user) {
    user = sc.loginUser;
  }
  return reqiredSetting.settings(user);
}



/**
 * Get one configuration defined in an Information Center setting.
 * @param {string} settingName - The name of the setting definition.
 * @param {string} name - The name of the configuration.
 * @param {object} [user] - Optional. The user context, the setting is evaluated. Current logged in user is default.
 * @returns {object} The first configuration defined in the requrested setting and that matched the name.
 */
function getConfiguration(settingName, name, user) {
  let settings = listConfigurations(settingName, user);
  return settings.find(configuration => (configuration.name === name));
}



/**
 * Exported functions provided externaly
 */
module.exports = {
  listSettings: listSettings,
  listConfigurations: listConfigurations,
  getConfiguration: getConfiguration
};
